document.addEventListener('DOMContentLoaded', () => {
    // --- DATOS DE EJEMPLO DE PRODUCTOS ---
    // Array que contiene la información detallada de cada producto.
    // Cada objeto de producto incluye su ID, nombre, descripción, categoría, código
    // y un array de URLs de imágenes para el slider.
    const products = [
        { id: 1, name: 'Silla Tiffany Blanca', description: 'Silla elegante y clásica, perfecta para bodas y eventos formal es.', category: 'Sillas', code: 'SITIF001', imageUrls: ['https://placehold.co/600x600/FFFFFF/333333?text=Silla+Tiffany+1', 'https://placehold.co/600x600/FAFAFA/333333?text=Silla+Tiffany+2', 'https://placehold.co/600x600/F0F0F0/333333?text=Silla+Tiffany+3'] },
        { id: 2, name: 'Mesa Redonda Imperial', description: 'Mesa de gran tamaño ideal para banquetes y reuniones, con capacidad para 10-12 personas.', category: 'Mesas', code: 'MESAIMP002', imageUrls: ['https://placehold.co/600x600/E0E0E0/333333?text=Mesa+Redonda+1', 'https://placehold.co/600x600/DADADA/333333?text=Mesa+Redonda+2'] },
        { id: 3, name: 'Carpa Elegante 10x5m', description: 'Carpa resistente y espaciosa, ideal para eventos al aire libre, protege del sol y la lluvia.', category: 'Carpas', code: 'CARPAELG003', imageUrls: ['https://placehold.co/600x600/F0F0F0/333333?text=Carpa+Elegante+1', 'https://placehold.co/600x600/E8E8E8/333333?text=Carpa+Elegante+2', 'https://placehold.co/600x600/F5F5F5/333333?text=Carpa+Elegante+3'] },
        { id: 4, name: 'Mantelería de Lino', description: 'Manteles de lino de alta calidad, disponibles en varios colores para complementar tu decoración.', category: 'Accesorios', code: 'MANTLINO004', imageUrls: ['https://placehold.co/600x600/DDDDDD/333333?text=Manteleria+Lino'] },
        { id: 5, name: 'Silla Avant Garde', description: 'Silla moderna y cómoda, perfecta para eventos contemporáneos y conferencias.', category: 'Sillas', code: 'SIAVG005', imageUrls: ['https://placehold.co/600x600/FFFFFF/333333?text=Silla+Avant+Garde+A', 'https://placehold.co/600x600/FAFAFA/333333?text=Silla+Avant+Garde+B'] },
        { id: 6, name: 'Mesa Rectangular de Madera', description: 'Mesa robusta de madera, ideal para cenas rústicas o estaciones de comida.', category: 'Mesas', code: 'MESAMAD006', imageUrls: ['https://placehold.co/600x600/D2B48C/FFFFFF?text=Mesa+Madera+1', 'https://placehold.co/600x600/C1A37E/FFFFFF?text=Mesa+Madera+2'] },
        { id: 7, name: 'Pista de Baile Iluminada', description: 'Pista de baile con luces LED integradas, crea un ambiente vibrante y divertido.', category: 'Pisos y Escenarios', code: 'PISTAB007', imageUrls: ['https://placehold.co/600x600/333333/FFFFFF?text=Pista+Baile+1', 'https://placehold.co/600x600/444444/FFFFFF?text=Pista+Baile+2', 'https://placehold.co/600x600/555555/FFFFFF?text=Pista+Baile+3'] },
        { id: 8, name: 'Centros de Mesa Florales', description: 'Arreglos florales personalizados para embellecer tus mesas y espacios.', category: 'Decoración', code: 'CMFLOR008', imageUrls: ['https://placehold.co/600x600/90EE90/333333?text=Flores+1'] },
        { id: 9, name: 'Iluminación Ambiental LED', description: 'Sistema de iluminación LED para crear ambientes únicos y destacar elementos decorativos.', category: 'Decoración', code: 'ILLUMLED009', imageUrls: ['https://placehold.co/600x600/ADD8E6/333333?text=Iluminacion+LED'] },
        { id: 10, name: 'Vajilla Premium', description: 'Juego de vajilla de porcelana fina, perfecto para cenas de gala y eventos exclusivos.', category: 'Accesorios', code: 'VAJILLAP010', imageUrls: ['https://placehold.co/600x600/F8F8F8/333333?text=Vajilla+1'] },
    ];

    // Array para almacenar los productos que el usuario ha añadido a su cotización.
    let quoteItems = [];
    // Variable para mantener el filtro de categoría actual ('Todos' por defecto).
    let currentFilter = 'Todos';

    // --- REFERENCIAS A ELEMENTOS DEL DOM ---
    // Se obtienen referencias a los elementos HTML que serán manipulados por JavaScript.
    const productGrid = document.getElementById('product-grid');
    const quoteCount = document.getElementById('quote-count');
    const quoteFormView = document.getElementById('quote-form-view');
    const viewQuoteBtn = document.getElementById('view-quote-btn');
    const backToCatalogBtn = document.getElementById('back-to-catalog-btn');
    const selectedProductsList = document.getElementById('selected-products-list');
    const emptyQuoteMessage = document.getElementById('empty-quote-message');
    const contactForm = document.getElementById('contact-form');
    const submitQuoteBtn = document.getElementById('submit-quote-btn');
    const submitIcon = document.getElementById('submit-icon');
    const submitText = document.getElementById('submit-text');
    // Se actualiza la referencia para el nuevo dropdown
    const categoryDropdown = document.getElementById('category-dropdown'); 
    const backToTopBtn = document.getElementById('back-to-top-btn');

    // Elementos para la caja de mensajes personalizada (reemplazo de alert()).
    const messageBoxBackdrop = document.getElementById('message-box-backdrop');
    const messageBoxTitle = document.getElementById('message-box-title');
    const messageBoxText = document.getElementById('message-box-text');
    const messageBoxOkBtn = document.getElementById('message-box-ok-btn');

    /**
     * Muestra una caja de mensajes personalizada en lugar de la alerta del navegador.
     * Esto mejora la experiencia de usuario y el control sobre el estilo.
     * @param {string} title - El título de la caja de mensajes.
     * @param {string} message - El contenido del mensaje.
     */
    function showMessageBox(title, message) {
        messageBoxTitle.textContent = title;
        messageBoxText.textContent = message;
        messageBoxBackdrop.classList.add('show');
    }

    // Event listener para el botón "Aceptar" en la caja de mensajes.
    // Al hacer clic, oculta la caja de mensajes.
    messageBoxOkBtn.addEventListener('click', () => {
        messageBoxBackdrop.classList.remove('show');
    });

    // --- LÓGICA DEL SLIDER DE IMÁGENES PARA CADA PRODUCTO ---
    /**
     * Configura la funcionalidad del slider de imágenes para un elemento de producto dado.
     * Esto incluye la navegación con flechas y los puntos indicadores, y ahora también el deslizamiento táctil.
     * @param {HTMLElement} sliderElement - El contenedor del slider de imágenes de un producto.
     */
    const setupSlider = (sliderElement) => {
        const wrapper = sliderElement.querySelector('.slider-wrapper');
        const dotsContainer = sliderElement.querySelector('.slider-dots');
        const prevBtn = sliderElement.querySelector('.slider-nav.prev');
        const nextBtn = sliderElement.querySelector('.slider-nav.next');
        const slides = wrapper.querySelectorAll('img');
        let currentIndex = 0;
        let startX = 0; // Para el inicio del toque
        let currentTranslate = 0; // Para el desplazamiento actual durante el toque
        let isDragging = false; // Bandera para saber si se está arrastrando

        // Función para actualizar la visualización del slider (posición y puntos activos).
        function updateSlider() {
            currentTranslate = -currentIndex * 100;
            wrapper.style.transform = `translateX(${currentTranslate}%)`;
            // Actualiza la clase 'active' en los puntos de navegación.
            if (dotsContainer) {
                dotsContainer.querySelectorAll('.dot').forEach((dot, index) => {
                    dot.classList.toggle('active', index === currentIndex);
                });
            }
        }

        // Event listener para la navegación por puntos.
        if (dotsContainer) {
            dotsContainer.addEventListener('click', (e) => {
                if (e.target.classList.contains('dot')) {
                    currentIndex = parseInt(e.target.dataset.index);
                    updateSlider();
                }
            });
        }

        // Event listener para el botón de navegación "anterior".
        if (prevBtn) {
            prevBtn.addEventListener('click', () => {
                currentIndex = (currentIndex > 0) ? currentIndex - 1 : slides.length - 1;
                updateSlider();
            });
        }

        // Event listener para el botón de navegación "siguiente".
        if (nextBtn) {
            nextBtn.addEventListener('click', () => {
                currentIndex = (currentIndex < slides.length - 1) ? currentIndex + 1 : 0;
                updateSlider();
            });
        }

        // --- Lógica para el deslizamiento (swipe) táctil ---
        sliderElement.addEventListener('touchstart', (e) => {
            startX = e.touches[0].clientX; // Guarda la posición X inicial del toque
            isDragging = true;
            wrapper.style.transition = 'none'; // Deshabilita la transición CSS durante el arrastre
        });

        sliderElement.addEventListener('touchmove', (e) => {
            if (!isDragging) return;
            const currentX = e.touches[0].clientX;
            const diffX = currentX - startX; // Diferencia de desplazamiento
            // Calcula la nueva posición del wrapper, permitiendo un arrastre visual
            wrapper.style.transform = `translateX(calc(${currentTranslate}% + ${diffX}px))`;
        });

        sliderElement.addEventListener('touchend', (e) => {
            if (!isDragging) return;
            isDragging = false;
            wrapper.style.transition = 'transform 0.3s ease-in-out'; // Vuelve a habilitar la transición

            const endX = e.changedTouches[0].clientX;
            const swipeDistance = endX - startX; // Distancia total del deslizamiento
            const threshold = 50; // Umbral en píxeles para considerar un deslizamiento

            if (swipeDistance > threshold) {
                // Deslizamiento hacia la derecha (anterior)
                currentIndex = (currentIndex > 0) ? currentIndex - 1 : slides.length - 1;
            } else if (swipeDistance < -threshold) {
                // Deslizamiento hacia la izquierda (siguiente)
                currentIndex = (currentIndex < slides.length - 1) ? currentIndex + 1 : 0;
            }
            updateSlider(); // Actualiza el slider a la nueva posición (o la misma si no hubo deslizamiento suficiente)
        });
    };

    /**
     * Renderiza la cuadrícula de productos basándose en el filtro de categoría activo.
     * Crea las tarjetas de producto dinámicamente y las inserta en el DOM.
     * @param {string} filter - La categoría a filtrar, o 'Todos' para mostrar todos los productos.
     */
    function renderProducts(filter = 'Todos') {
        productGrid.innerHTML = ''; // Limpia los productos existentes en la cuadrícula.
        // Filtra los productos según la categoría seleccionada.
        const filteredProducts = filter === 'Todos' ? products : products.filter(p => p.category === filter);

        // Muestra un mensaje si no hay productos en la categoría seleccionada.
        if (filteredProducts.length === 0) {
            productGrid.innerHTML = '<p class="text-center text-gray-500 col-span-full py-10">No hay productos en esta categoría.</p>';
            return;
        }

        // Itera sobre los productos filtrados para crear y añadir sus tarjetas al DOM.
        filteredProducts.forEach(product => {
            // Verifica si el producto ya está en la cotización para actualizar el botón.
            const isAdded = quoteItems.some(item => item.id === product.id);
            const buttonText = isAdded ? 'Añadido' : 'Añadir a Cotización';
            const buttonClasses = isAdded ? 'added' : '';

            // Genera el HTML para las imágenes del slider y los puntos indicadores.
            // Se incluye un manejador de errores para las imágenes.
            const imagesHtml = product.imageUrls.map(url => `<img src="${url}" alt="${product.name}" class="w-full h-full object-cover" onerror="this.onerror=null;this.src='https://placehold.co/600x600/CCCCCC/666666?text=Imagen+No+Disponible';">`).join('');
            const dotsHtml = product.imageUrls.length > 1 ? product.imageUrls.map((_, index) => `<div class="dot ${index === 0 ? 'active' : ''}" data-index="${index}"></div>`).join('') : '';

            const productCard = document.createElement('div');
            productCard.className = 'product-card fade-in'; // Añade animación de aparición.
            productCard.innerHTML = `
                <div class="product-image-slider">
                    <div class="slider-wrapper">${imagesHtml}</div>
                    ${product.imageUrls.length > 1 ? `
                        <div class="slider-nav prev"><i class="fa-solid fa-chevron-left"></i></div>
                        <div class="slider-nav next"><i class="fa-solid fa-chevron-right"></i></div>
                        <div class="slider-dots">${dotsHtml}</div>
                    ` : ''}
                </div>
                <div class="p-4 flex flex-col flex-grow">
                    <h3 class="title mb-2">${product.name}</h3>
                    <p class="text-sm text-gray-700 mb-2">${product.description}</p>
                    <p class="text-sm font-semibold text-brand-dark-blue mb-1 flex items-center justify-center">
                        <i class="fa-solid fa-tag mr-2 text-brand-red-accent"></i>Categoría: ${product.category}
                    </p>
                    <p class="text-sm font-semibold text-brand-dark-blue mb-4 flex items-center justify-center">
                        <i class="fa-solid fa-barcode mr-2 text-brand-red-accent"></i>Código: ${product.code}
                    </p>
                    <button data-product-id="${product.id}" class="btn-action btn-add-quote w-full mt-auto ${buttonClasses}" ${isAdded ? 'disabled' : ''}>
                        ${buttonText}
                    </button>
                </div>
            `;
            productGrid.appendChild(productCard);

            // Inicializa el slider para esta tarjeta si tiene múltiples imágenes.
            if (product.imageUrls.length > 1) {
                setupSlider(productCard.querySelector('.product-image-slider'));
            }
        });
    }

    /**
     * Renderiza las opciones del menú desplegable de categoría.
     * Genera una opción para cada categoría única de productos, más una opción "Todos".
     */
    function renderCategoryDropdown() {
        // Obtiene las categorías únicas de los productos y añade 'Todos'.
        const categories = ['Todos', ...new Set(products.map(p => p.category))];
        categoryDropdown.innerHTML = ''; // Limpia las opciones existentes.
        categories.forEach(category => {
            const option = document.createElement('option');
            option.value = category;
            option.textContent = category;
            // Marca la opción como seleccionada si coincide con el filtro actual.
            if (category === currentFilter) {
                option.selected = true;
            }
            categoryDropdown.appendChild(option);
        });
    }

    /**
     * Actualiza el contador de productos en la cotización y la visibilidad del botón "Mi Cotización".
     */
    function updateQuoteCount() {
        quoteCount.textContent = quoteItems.length;
        // Muestra u oculta el botón "Mi Cotización" según si hay productos en la lista.
        if (quoteItems.length > 0) {
            viewQuoteBtn.classList.remove('hidden');
        } else {
            viewQuoteBtn.classList.add('hidden'); 
        }
    }

    /**
     * Muestra el panel lateral del formulario de solicitud de cotización.
     * También renderiza el resumen de la cotización dentro del panel.
     */
    function showQuoteForm() {
        quoteFormView.classList.add('show');
        renderQuoteSummary(); // Renderiza el resumen de la cotización al abrir el panel.
    }

    /**
     * Oculta el panel lateral del formulario de solicitud de cotización.
     */
    function hideQuoteForm() {
        quoteFormView.classList.remove('show');
    }
    
    /**
     * Renderiza la lista de productos seleccionados en el resumen de la cotización dentro del panel lateral.
     * Muestra un mensaje si la cotización está vacía.
     */
    function renderQuoteSummary() {
        selectedProductsList.innerHTML = ''; // Limpia la lista existente.
        // Si no hay productos en la cotización, muestra un mensaje y deshabilita el botón de enviar.
        if (quoteItems.length === 0) {
            emptyQuoteMessage.classList.remove('hidden');
            submitQuoteBtn.disabled = true;
        } else {
            emptyQuoteMessage.classList.add('hidden');
            submitQuoteBtn.disabled = false;
            // Itera sobre los productos en la cotización para mostrarlos en el resumen.
            quoteItems.forEach(item => {
                const listItem = document.createElement('div');
                listItem.className = 'flex items-center justify-between bg-gray-50 p-3 rounded-md shadow-sm';
                listItem.innerHTML = `
                    <div class="flex items-center">
                        <img src="${item.imageUrls[0]}" alt="${item.name}" class="w-16 h-16 object-cover rounded-md mr-4" onerror="this.onerror=null;this.src='https://placehold.co/600x600/CCCCCC/666666?text=Imagen+No+Disponible';">
                        <div>
                            <p class="font-semibold text-brand-dark-blue">${item.name}</p>
                            <p class="text-sm text-gray-500 flex items-center"><i class="fa-solid fa-tag mr-2 text-brand-red-accent"></i>Categoría: ${item.category}</p>
                            <p class="text-sm text-gray-500 flex items-center"><i class="fa-solid fa-barcode mr-2 text-brand-red-accent"></i>Código: ${item.code}</p>
                        </div>
                    </div>
                    <button data-remove-id="${item.id}" class="font-semibold text-sm hover:underline" style="color: var(--brand-red-accent);">Quitar</button>
                `;
                selectedProductsList.appendChild(listItem);
            });
        }
    }
    
    // --- GESTIÓN DE EVENTOS ---

    // Listener para añadir productos a la cotización desde la cuadrícula de productos.
    // Utiliza delegación de eventos para manejar clics en los botones "Añadir a Cotización".
    productGrid.addEventListener('click', (e) => {
        const button = e.target.closest('.btn-add-quote');
        if (button && !button.disabled) {
            const productId = parseInt(button.dataset.productId);
            const product = products.find(p => p.id === productId);
            if (product) {
                quoteItems.push(product);
                updateQuoteCount(); // Actualiza el contador de la cotización.
                renderProducts(currentFilter); // Vuelve a renderizar para actualizar el estado del botón.
            }
        }
    });

    // Listener para quitar productos del resumen de la cotización en el panel lateral.
    // Utiliza delegación de eventos para manejar clics en los botones "Quitar".
    selectedProductsList.addEventListener('click', (e) => {
        if(e.target.dataset.removeId) {
            const productId = parseInt(e.target.dataset.removeId);
            // Filtra el array para eliminar el producto con el ID correspondiente.
            quoteItems = quoteItems.filter(item => item.id !== productId);
            renderQuoteSummary(); // Vuelve a renderizar el resumen.
            updateQuoteCount(); // Actualiza el contador.
            renderProducts(currentFilter); // Vuelve a renderizar los productos para restablecer los botones "Añadido".
        }
    });

    // Listener para el menú desplegable de categoría.
    // Actualiza el filtro actual y renderiza los productos.
    categoryDropdown.addEventListener('change', (e) => {
        currentFilter = e.target.value; // Establece el filtro actual desde el valor seleccionado.
        renderProducts(currentFilter); // Renderiza los productos con el nuevo filtro.
    });

    // Listeners para mostrar y ocultar el formulario de cotización.
    viewQuoteBtn.addEventListener('click', showQuoteForm);
    backToCatalogBtn.addEventListener('click', hideQuoteForm);
    // Cierra el panel si se hace clic en el fondo oscuro (fuera del contenido del panel).
    quoteFormView.addEventListener('click', (e) => {
        if (e.target === quoteFormView) {
            hideQuoteForm();
        }
    });

    // Lógica para mostrar/ocultar el botón "Volver Arriba" al hacer scroll.
    window.addEventListener('scroll', () => {
        if (window.scrollY > 300) { // Muestra el botón después de desplazar 300px.
            backToTopBtn.classList.add('show');
        } else {
            backToTopBtn.classList.remove('show');
        }
    });

    // Listener para el botón "Volver Arriba" para desplazar la página al inicio.
    backToTopBtn.addEventListener('click', () => {
        window.scrollTo({ top: 0, behavior: 'smooth' }); // Desplazamiento suave.
    });

    // Event listener para el envío del formulario de contacto.
    // Realiza validaciones y simula el envío de datos.
    contactForm.addEventListener('submit', async (e) => {
        e.preventDefault(); // Previene el envío por defecto del formulario.

        // Validación básica de campos obligatorios.
        const name = document.getElementById('name').value.trim();
        const email = document.getElementById('email').value.trim();
        const eventDate = document.getElementById('event-date').value.trim();

        if (!name || !email || !eventDate) {
            showMessageBox('Error de Formulario', 'Por favor, completa todos los campos obligatorios (Nombre, Correo Electrónico, Fecha del Evento).');
            return;
        }

        // Validación simple del formato de correo electrónico usando una expresión regular.
        const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailPattern.test(email)) {
            showMessageBox('Error de Formulario', 'Por favor, introduce un formato de correo electrónico válido.');
            return;
        }

        // Verifica si hay productos en la cotización antes de permitir el envío.
        if (quoteItems.length === 0) {
            showMessageBox('Error de Cotización', 'Por favor, añade al menos un producto a tu cotización antes de enviarla.');
            return;
        }

        // Muestra el estado de carga en el botón de envío.
        submitQuoteBtn.disabled = true;
        submitText.textContent = 'Enviando...';
        submitIcon.classList.remove('fa-paper-plane');
        submitIcon.classList.add('fa-spinner', 'fa-spin');

        // Simula una llamada a la API con un retardo para mostrar el estado de carga.
        await new Promise(resolve => setTimeout(resolve, 2000)); 

        // Muestra los datos de la solicitud en la consola (en un entorno real, estos datos
        // se enviarían a un servidor a través de una petición fetch/XMLHttpRequest).
        console.log('--- NUEVA SOLICITUD DE COTIZACIÓN ---');
        console.log('Datos del Cliente:', {
            name: document.getElementById('name').value,
            email: document.getElementById('email').value,
            phone: document.getElementById('phone').value,
            eventDate: document.getElementById('event-date').value,
            comments: document.getElementById('comments').value
        });
        console.log('Productos Solicitados:', quoteItems);

        // Muestra un mensaje de éxito al usuario.
        showMessageBox('¡Solicitud Enviada!', 'Tu solicitud de cotización ha sido recibida con éxito. Nos pondremos en contacto contigo pronto.');
        
        hideQuoteForm(); // Oculta el panel del formulario.
        quoteItems = []; // Limpia la cotización después del envío exitoso.
        contactForm.reset(); // Reinicia los campos del formulario.
        updateQuoteCount(); // Actualiza el contador de la cotización a 0.
        renderProducts(currentFilter); // Vuelve a renderizar los productos para restablecer los botones "Añadido".

        // Restablece el estado del botón de envío a su estado original.
        submitQuoteBtn.disabled = false;
        submitText.textContent = 'Enviar Solicitud';
        submitIcon.classList.remove('fa-spinner', 'fa-spin');
        submitIcon.classList.add('fa-paper-plane');
    });

    // --- LLAMADAS DE RENDERIZADO INICIALES ---
    // Estas funciones se ejecutan cuando el DOM ha sido completamente cargado
    // para inicializar la interfaz de usuario.
    renderCategoryDropdown(); // Muestra las opciones del menú desplegable de categoría.
    renderProducts(currentFilter); // Muestra todos los productos inicialmente.
    updateQuoteCount(); // Inicializa el contador de la cotización (y oculta el botón si está en 0).
});